using System.Collections;
using System.Collections.Generic;
using Unity.VisualScripting;
using UnityEngine;
using TMPro;
using System.Text.RegularExpressions;
using UnityEditor;
using UnityEngine.UIElements.Experimental;
using System;
using Unity.VisualScripting.Dependencies.NCalc;
using System.Runtime.CompilerServices;

namespace PokerTable
{
    public class ActionList : MonoBehaviour
    {
        public List<Action> Actions;
        public float dt;
        public bool isPause = false;
        void Awake()
        {
            Actions = new List<Action>(); // Initialize
        }
        void Update()
        {
            dt = isPause ? 0.0f : Time.deltaTime * Time.timeScale;
            
            int BlockingGroupSet = 0;
            for (int i = 0; i < Actions.Count; i++)
            {
                if (Actions[i].IsInGroupSet(BlockingGroupSet))
                    continue;
                if (Actions[i].IncrementTime(dt) == false)
                    continue;
                if (Actions[i].Update(dt) == false)
                {
                    Actions.RemoveAt(i);
                    i--;
                    continue;
                }
                if (Actions[i].Blocking == true)
                    BlockingGroupSet = BlockingGroupSet | Actions[i].Groups;
            }
        }
        public Action TranslateObject(GameObject objectToMove, Vector3 start, Vector3 end, float duration = 0.0f, float delay = 0.0f, bool useLocal = false)
        {
            Translate addedAction = new(objectToMove, start, end, duration, delay);
            Actions.Add(addedAction);
            return addedAction;
        }
        public Action TranslateObject(GameObject objectToMove, Vector3 end, float duration = 0.0f, float delay = 0.0f, int group = 0, Action.EaseType ease = Action.EaseType.Linear, bool useLocal = false) //Vector3
        {
            Vector3 start = useLocal ? objectToMove.transform.localPosition : objectToMove.transform.position;
            Translate addedAction = new(objectToMove, objectToMove.transform.position, end, duration, delay, group, ease, useLocal);
            Actions.Add(addedAction);
            return addedAction;
        }
        public Action TranslateObject(GameObject objectToMove, Vector2 end, float duration = 0.0f, float delay = 0.0f, bool useLocal = false) //Vector2
        {
            Translate addedAction = new(objectToMove, objectToMove.transform.position, end, duration, delay);
            Actions.Add(addedAction);
            return addedAction;
        }
        public Action RotateObject(GameObject objectToRotate, Vector3 start, Vector3 end, float duration = 0.0f, float delay = 0.0f)
        {
            Rotate addedAction = new(objectToRotate, start, end, duration, delay);
            Actions.Add(addedAction);
            return addedAction;
        }
        public Action RotateObjectAbsolute(GameObject objectToRotate, Vector3 end, float duration = 0.0f, float delay = 0.0f, int group = 0, Action.EaseType ease = Action.EaseType.Linear, bool useLocal = false)
        {
            Rotate addedAction = new(objectToRotate,objectToRotate.transform.localEulerAngles, end, duration, delay, group, ease, useLocal);
            Actions.Add(addedAction);
            return addedAction;
        } 
        public Action RotateObjectRelative(GameObject objectToRotate, Vector3 end, float duration = 0.0f, float delay = 0.0f, int group = 0, Action.EaseType ease = Action.EaseType.Linear, bool useLocal = false)
        {
            Rotate addedAction = new(objectToRotate, objectToRotate.transform.localEulerAngles, objectToRotate.transform.localEulerAngles + end, duration, delay);
            Actions.Add(addedAction);
            return addedAction;
        } 
        public Action ScaleObject(GameObject objectToScale, Vector3 start, Vector3 end, float duration = 0.0f, float delay = 0.0f, int group = 0, Action.EaseType ease = Action.EaseType.Linear)
        {
            Scale addedAction = new(objectToScale, start, end, duration, delay);
            Actions.Add(addedAction);
            return addedAction;
        } 
        public Action ScaleObject(GameObject objectToScale, Vector3 end, float duration = 0.0f, float delay = 0.0f, int group = 0, Action.EaseType ease = Action.EaseType.Linear)
        {
            Scale addedAction = new(objectToScale, objectToScale.transform.localScale, end, duration, delay, group, ease);
            Actions.Add(addedAction);
            return addedAction;
        } 
        public Action FadeObject(GameObject objectToFade, Color start, Color end, float duration = 0.0f, float delay = 0.0f, bool isTMPro = false, int group = 0, Action.EaseType ease = Action.EaseType.Linear)
        {
            Fade addedAction = new(objectToFade, start, end, duration, delay, isTMPro, group, ease);
            Actions.Add(addedAction);
            return addedAction;
        }
        public Action FadeObject(GameObject objectToFade, Color end, float duration = 0.0f, float delay = 0.0f, bool isTMPro = false, int group = 0, Action.EaseType ease = Action.EaseType.Linear)
        {
            Fade addedAction;
            if(isTMPro)
            {
                addedAction = new(objectToFade, objectToFade.GetComponent<TextMeshPro>().color, end, duration, delay, isTMPro, group, ease);
            }
            else
            {
                addedAction = new(objectToFade, objectToFade.GetComponentInChildren<SpriteRenderer>().color, end, duration, delay, isTMPro, group, ease);
            }
            Actions.Add(addedAction);
            return addedAction;
        } 
        public Action DeleteObject(GameObject objectToDelete, float delay = 0.0f, int group = 0)
        {
            Delete addedAction = new(objectToDelete, delay);
            Actions.Add(addedAction);
            return addedAction;
        }
        public Action CallFunction(System.Action function, float delay = 0.0f)
        {
            Call addedAction = new(function, delay);
            Actions.Add(addedAction);
            return addedAction;
        }
        public void ReverseAll()
        {
            foreach (Action action in Actions)
            {
                action.Reverse();
            }
        }
    }

    public class Action
    {
        public float Delay;
        public float TimePassed;
        public float Duration;
        public float PercentDone;
        public bool Blocking;
        public int Groups = 0;
        public EaseType Easing;
        public GameObject ActionObject = null;
        public enum EaseType
        {
            Linear,
            EaseIn,
            EaseOut,
            FastIn,
            FastOut,
            InAndOut,
            Bounce,
        };
        public float Ease(float percent, EaseType type)
        {
            if (percent <= 0.0f)
                return 0.0f;
            if (percent >= 1.0f)
                return 1.0f;
            switch (type)
            {
                case EaseType.Linear: return percent;
                case EaseType.EaseIn: return Mathf.Sqrt(percent); // Fast -> Slow
                case EaseType.EaseOut: return Mathf.Pow(percent, 2.0f); // Slow -> Fast
                case EaseType.FastIn: return Mathf.Sqrt(Mathf.Sqrt(percent)); // Fast -> Slow
                case EaseType.FastOut: return Mathf.Pow(percent, 3.0f); // Slow -> Fast
                case EaseType.InAndOut: 
                    if (percent < 0.5f)
                        return Mathf.Pow(percent * 2.0f, 2.0f) * 0.5f;
                    else
                        return Mathf.Sqrt((percent - 0.5f)*2.0f) * 0.5f + 0.5f;
                case EaseType.Bounce:
                    float n = 7.5625f;
                    float d = 2.75f;
                    if (percent < 1.0f / d)
                        return n * percent * percent;
                    else if (percent < 2.0f / d)
                        return n * (percent -= 1.5f / d) * percent + 0.75f;
                    else if (percent < 2.5f / d)
                        return n * (percent -= 2.25f / d) * percent + 0.9375f;
                    else
                        return n * (percent -= 2.625f / d) * percent + 0.984375f;
            }
            return percent;
        }
        public virtual bool Update(float dt)
        {
            return PercentDone < 1.0f;
        }
        public bool IncrementTime(float dt)
        {
            // If there is a delay Pause, and decrease it.
            if (Delay > 0.0f)
            {
                Delay -= dt;
                if (Delay > 0.0f) // If delay is exist, pause
                    return false;
                TimePassed -= Delay;
                Delay = 0.0f;
            }
            else
            {
                TimePassed += dt;
            }

            if (TimePassed >= Duration)
            {
                TimePassed = Duration;
                PercentDone = 1.0f;
            }
            else
            {
                PercentDone = TimePassed / Duration;
            }

            PercentDone = Ease(PercentDone,Easing);

            return true;
        }
        public void AddToGroup(int groupNum)
        {
            if (groupNum < 1 || groupNum > 30) return;
            Groups = Groups | (1 << (groupNum - 1));
        }
        public bool IsInGroup (int groupNum)
        {
            if (groupNum < 1 || groupNum > 30) return false;
            return (Groups & (1 << (groupNum -1))) != 0;
        }
        public bool IsInGroupSet(int groupSet)
        {
            if (groupSet < 0) return false;
            return (Groups & groupSet) != 0;
        }
        public void ReverseTime()
        {
            if(Delay > 0)
                return;
            TimePassed = Duration - TimePassed;
            PercentDone = TimePassed / Duration;
            
        }
        public virtual void Reverse()
        {  
            ReverseTime();
        }
    }
    public class Translate : Action
    {
        Vector3 Start;
        Vector3 End;
        bool UseLocal;

        public Translate(GameObject objectToMove, Vector3 start, Vector3 end, float duration = 0.0f, float delay = 0.0f, int group = 0, EaseType ease = EaseType.Linear, bool useLocal = false)
        {
            ActionObject = objectToMove;
            Start = start;
            End = end;
            Duration = duration;
            Delay = delay;
            AddToGroup(group);
            Easing = ease;
            UseLocal = useLocal;
        }

        public override bool Update(float dt)
        {
            if (ActionObject == null)
                return false;
            if (UseLocal)
            {
                ActionObject.transform.localPosition = Start + (End - Start) * PercentDone;
            }
            else
            {
                ActionObject.transform.position = Start + (End - Start) * PercentDone;
            }
            return PercentDone < 1.0f;
        }

        public override void Reverse()
        {
            if(Delay > 0)
                return;
            (Start, End) = (End, Start);
            ReverseTime();
        }
    }
    public class Rotate : Action
    {
        Vector3 Start;
        Vector3 End;
        bool UseLocal;
        public Rotate(GameObject objectToRotate, Vector3 start, Vector3 end, float duration = 0.0f, float delay = 0.0f, int group = 0, EaseType ease = EaseType.Linear, bool useLocal = false)
        {
            ActionObject = objectToRotate;
            Start = start;
            End = end;
            Duration = duration;
            Delay = delay;
            AddToGroup(group);
            Easing = ease;
            UseLocal = useLocal;
        }
        public override bool Update(float dt)
        {
            if (ActionObject == null)
                return false;
            // if (UseLocal)
                 ActionObject.transform.localEulerAngles = Start + (End - Start) * PercentDone;
            // else
            //     ActionObject.transform.eulerAngles = Start + (End - Start) * PercentDone;

            return PercentDone < 1.0f;
        }
    }
    public class Scale : Action
    {
        Vector3 Start;
        Vector3 End;
        public Scale(GameObject objectToScale, Vector3 start, Vector3 end, float duration = 0.0f, float delay = 0.0f, int group = 0, EaseType ease = EaseType.Linear)
        {
            ActionObject = objectToScale;
            Start = start;
            End = end;
            Duration = duration;
            Delay = delay;
            AddToGroup(group);
            Easing = ease;
        }
        public override bool Update(float dt)
        {
            if (ActionObject == null)
                return false;
            ActionObject.transform.localScale = Start + (End - Start) * PercentDone;
            return PercentDone < 1.0f;
        }
    }
    public class Fade : Action
    {
        Color StartColor;
        Color EndColor;
        bool IsTMPro;
        public Fade(GameObject objectToFade, Color startColor, Color endColor, float duration = 0.0f, float delay = 0.0f, bool isTMPro = false, int group = 0, EaseType ease = EaseType.Linear)
        {
            ActionObject = objectToFade;
            StartColor = startColor;
            EndColor = endColor;
            Duration = duration;
            Delay = delay;
            IsTMPro = isTMPro;
            AddToGroup(group);
            Easing = ease;
        }
        public override bool Update(float dt)
        {
            if (ActionObject == null)
                return false;
            if (IsTMPro)
            {
                ActionObject.GetComponent<TextMeshPro>().color = StartColor + (EndColor - StartColor) * PercentDone;
            }
            else
            {
                SpriteRenderer[] spriteRenderers;
                spriteRenderers = ActionObject.GetComponentsInChildren<SpriteRenderer>();
                foreach (SpriteRenderer sprites in spriteRenderers)
                {
                    sprites.color = StartColor + (EndColor - StartColor) * PercentDone;
                }
            }
            return PercentDone < 1.0f;
        }
    }
    public class Delete : Action
    {
        public Delete(GameObject objectToDelete, float delay = 0.0f, int group = 0)
        {
            ActionObject = objectToDelete;
            Duration = 0.0f;
            Delay = delay;
            AddToGroup(group);
        }
        public override bool Update(float dt)
        {
            if (ActionObject == null)
                return false;
            UnityEngine.Object.Destroy(ActionObject);
            return PercentDone < 1.0f;
        }
    }
    public class Call : Action
    {
        System.Action FunctionToCall;
        public Call(System.Action function, float delay = 0.0f)
        {
            FunctionToCall = function;
            Duration = 0.0f;
            Delay = delay;
        }
        public override bool Update(float dt)
        {
               FunctionToCall();
               return PercentDone < 1.0f;
        }
    }
}
