using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using System.Linq;

namespace PokerTable{

public class StaticData : MonoBehaviour
{
    private enum HandRank
    {
        HighCard = 1,
        OnePair,
        TwoPair,
        ThreeOfAKind,
        FourOfAKind
    }    
    public static Vector3 EvaluateHands(List<GameObject> hands)
    {
        if (hands == null || hands.Count == 0)
        {
            return Vector3.zero;
        }
        // Extract card values and suits
        var cardValues = hands.Select(hand => (int)hand.GetComponent<Card>().v2_value.x).ToList();
        var cardSuits = hands.Select(hand => (int)hand.GetComponent<Card>().v2_value.y).ToList();

        if (!cardValues.Any() || !cardSuits.Any())
        {
            // Handle the case where cardValues or cardSuits are empty
            return Vector3.zero; // Or throw a descriptive exception
        }
        
        // Count occurrences of each card value
        var valueCounts = cardValues.GroupBy(v => v).ToDictionary(group => group.Key, group => group.Count());

        // Find the highest frequency of any value (for identifying pairs, three/four of a kind)
        int maxCount = valueCounts.Max(entry => entry.Value);
        HandRank rank = DetermineHandRank(valueCounts);

        // Determine the value of the hand (highest value card in the hand or the value of the pair/triple/quadruple)
        int handValue = valueCounts.OrderByDescending(entry => entry.Value)
                                .ThenByDescending(entry => entry.Key)
                                .First().Key;

        // Assuming suit values are ordered and higher is better, similar to card values
        int handSuit = cardSuits.Distinct().OrderByDescending(s => s).First();

        return new Vector3((int)rank, handValue, handSuit);
    }

    private static HandRank DetermineHandRank(Dictionary<int, int> valueCounts)
    {
        int maxCount = valueCounts.Max(entry => entry.Value);
        if (maxCount == 4)
        {
            return HandRank.FourOfAKind;
        }
        else if (maxCount == 3)
        {
            return HandRank.ThreeOfAKind;
        }
        else if (maxCount == 2)
        {
            int pairCount = valueCounts.Count(entry => entry.Value == 2);
            return pairCount == 2 ? HandRank.TwoPair : HandRank.OnePair;
        }
        return HandRank.HighCard;
    }
    public static Sprite getCardImage(Vector2 cardVal) 
    {
        string suit = string.Empty;
        switch((int)cardVal.y)
        {
            case 1: suit = "a"; break;
            case 2: suit = "b"; break;
            case 3: suit = "c"; break;
            case 4: suit = "d"; break;
        }
        string path = $"Cards3/{(int)cardVal.x}{suit}";
        Sprite cardSprite = Resources.Load<Sprite>(path);
        if (cardSprite == null)
        {
            Debug.LogError($"Failed to load sprite at path: {path}");
        }
        return cardSprite;
    }
}

}//namespace PokerTable