using System.Collections;
using System.Collections.Generic;
using Unity.VisualScripting;
using UnityEditor;
using UnityEngine;
using System.Linq;
using TMPro;
using System.Diagnostics.SymbolStore;
using System.IO;
using UnityEngine.UIElements;

namespace PokerTable{
public class GameManager : MonoBehaviour
{
    [HideInInspector]
    public float nearCameraZ = -10.0f + 0.3f;
    [HideInInspector]
    public float ZDepthTable = -10.0f + 1000.0f -1.01f;
    // [HideInInspector]
    public bool isUIon = false;

    public UIManager uiManager;
    public ActionList buttonActionList;
    // game states
    [HideInInspector]
    public byte gameState = STATE_IDLE;
    public static byte STATE_IDLE = 0;  // idle state: waiting for player to press BET or FOLD
    public static byte STATE_PLAYER_TURN = 1;  // idle state: waiting for player to press BET or FOLD
    public static byte STATE_AI_TURN = 2;
    public static byte STATE_PLAYER_FOLD = 3;
    public static byte STATE_NEW_ROUND = 4; 
    public static byte STATE_NEW_GAME = 5; 
    public GameObject CardPrefab; // Put the card prefab in here.
    public GameObject ChipPrefab; // Put the chip prefab in here.
    public int NumberPerSuit; // Amount of cards per suit.
    public int ChipPerPlayer; // Amount of chips which each player have initially.
    public List<GameObject> CardDeck = new(); // Deck for Cards.
    public List<GameObject> AllCards = new();
    public List<GameObject> ChipPot = new(); // Deck for Chips.
    [Range (1,5)]
    public int AIPlayersNumber;
    public Vector3 CardDeckPosition; // Main deck position
    public ActionList actionList; // Reference to the ActionList component.
    [Header("Players")]
    public List<GameObject> playerList = new();
    public Color GizmoColor; // Color of Gizmo for player position
    public GameObject Player1;
    public GameObject Player2;
    public GameObject Player3;
    public GameObject Player4;
    public GameObject Player5;
    public GameObject Player6;
    public int currentPlayerIndex = -1; // Start before the first player
    
    public Color Player1Color = Color.blue; // Deafult player color
    public Color Player2Color = Color.yellow;
    public Color Player3Color = Color.magenta;
    public Color Player4Color = Color.cyan;
    public Color Player5Color = Color.red;
    public Color Player6Color = Color.white;

    public static int cardsTranslateGroup = 1;
    public static int cardsRotateGroup = 2;
    public static int chipsTranslateGroup = 3;

    public static float deckFadeInTime = 1.0f;
    public static float delayBetweenRounds = 3.0f;
    public static float shuffleTime = 1.0f;
    public static float cardThrowTime = 1.1f;

    public bool IsShuffled = false;
    public bool IsRevealing = false;
    public static StreamWriter DataStream;
    private int advanceTurnAttempts = 0;
    public bool isAutoMode = false;
    public int cycleCount = 0;
    private bool isProcessingEndOfRound = false;
    Vector3 GetRandomScaler(){
        return new Vector3(UnityEngine.Random.Range(1.0f, 3.0f), UnityEngine.Random.Range(1.0f, 3.0f), 1.0f);        
    }
    Vector3 GetRandomAngle(){
        return new Vector3(UnityEngine.Random.Range(0.0f, 180.0f),UnityEngine.Random.Range(0.0f, 180.0f),UnityEngine.Random.Range(0.0f, 180.0f));
    }
    public static Vector3 GetFlipAngle(){
        return new Vector3(0.0f, 180.0f, 0.0f);
    }
    Color GetRandomColor(){
        return new Color(UnityEngine.Random.Range(0.0f, 1.0f),UnityEngine.Random.Range(0.0f, 1.0f),UnityEngine.Random.Range(0.0f, 1.0f));
    }
    Vector3 GetRandomPositionOnScreen()
    {
        float minX = 0.0f;
        float maxX = 1.0f;
        float minY = 0.0f;
        float maxY = 1.0f;
        float minZ = -9.6f; // Based on the Main Camera position
        float maxZ = 900.0f; // Based on the Main Camera Clipping Planes

        // Get the screen bounds in world coordinates
        Vector3 minScreenBounds = Camera.main.ViewportToWorldPoint(new Vector3(minX, minY, 0));
        Vector3 maxScreenBounds = Camera.main.ViewportToWorldPoint(new Vector3(maxX, maxY, 0));

        // Generate random coordinates within the screen bounds
        return new Vector3(UnityEngine.Random.Range(minScreenBounds.x, maxScreenBounds.x), UnityEngine.Random.Range(minScreenBounds.y, maxScreenBounds.y), UnityEngine.Random.Range(minZ, maxZ));
    }
    // Shuffle any List<T>
    public void Shuffle<T>(List<T> list)
    {
        System.Random random = new();
        int n = list.Count;
        for (int i = n - 1; i > 0; i--)
        {
            int j = random.Next(i + 1);
            (list[i], list[j]) = (list[j], list[i]);
        }
    }
    // Generate Decks for cards, and Chips. Changable in InintialCardDeckCount, and InitialChipDeckCount from hierchy.
    public bool InitializeDeck()
    {
        CardDeck.Clear();
        for (int suit = 1; suit <= 4; suit++) // Loop through suits
        {
            for (int value = 1; value <= 13; value++) // Loop through values
            {
                // Instantiate a new card GameObject
                GameObject newCard = Instantiate(CardPrefab, new Vector3 (transform.position.x, transform.position.y, ZDepthTable), Quaternion.identity, gameObject.transform);
                // Set the card's value and suit
                Card cardComponent = newCard.GetComponent<Card>();
                if (cardComponent != null)
                {
                    cardComponent.v2_value = new Vector2(value, suit);
                    cardComponent.UpdateCardFaceSprite(cardComponent.v2_value);
                }
                // Add the new card to the deck
                // CardDeck.Add(newCard);
                AllCards.Add(newCard);
                SpriteRenderer[] spriteRenderers;
                spriteRenderers = newCard.GetComponentsInChildren<SpriteRenderer>();
                foreach (SpriteRenderer sprites in spriteRenderers)
                    sprites.color = Color.clear;
            }
        }
        // Fade in just single card
        foreach (var card in AllCards)
        {
            actionList.FadeObject(card, Color.clear, Color.white, deckFadeInTime, 0.0f, false, cardsTranslateGroup, Action.EaseType.EaseOut);
            CardDeck.Add(card);
        }
        Debug.Log("Deck Initialized: " + AllCards.Count);
        return true;
    }

    // Shuffle cards in scene. Collect all of the card to the deck list, and shuffle the order.
    public void ShuffleCards(float delay = 0.0f) // Do after decent time. (To show hands)
    {
        CardDeck.Clear();
        IsRevealing = false;
        IsShuffled = true;
        actionList.CallFunction(() =>
        {
            foreach(var card in AllCards)
            {
                CardDeck.Add(card);
                actionList.TranslateObject(card, new Vector3 (transform.position.x, transform.position.y, ZDepthTable), shuffleTime, 0.0f, cardsTranslateGroup, Action.EaseType.Linear);
                actionList.RotateObjectAbsolute(card, Vector3.zero, shuffleTime, 0.0f);
            }
            Shuffle(CardDeck);
        }, delay);
    }

    // Deal Cards to each players.
    // Condition Check: Read all list, Player receive card only when IsIn && !IsFold
    // If dealt player is NOT an AI, shows card face.
    void DealCards(float delay = 0.0f)
    {
        actionList.CallFunction(() =>
        {
            for(int i = 0; i < playerList.Count; i++)
            {
                if(CardDeck.Count == 0)
                {
                    return;
                }
                GameObject card = CardDeck[^1];
                GameObject playerObject = playerList[i];
                Player player = playerList[i].GetComponent<Player>();
                if(player.IsIn && !player.IsFold)
                {
                    Vector3 localOffset = new ((float)player.hand.Count * Player.cardGap, 0, 0);
                    Vector3 worldOffset = playerObject.transform.TransformPoint(localOffset) - playerObject.transform.position;
                        actionList.TranslateObject(card, playerObject.GetComponent<Transform>().position + worldOffset, cardThrowTime, 0.0f, cardsTranslateGroup, Action.EaseType.EaseIn);
                        if(!player.IsAI)
                        {
                            actionList.RotateObjectAbsolute(card, GetFlipAngle(), 1.0f, 0.0f, cardsRotateGroup);
                        }
                        else
                            actionList.RotateObjectAbsolute(card, playerObject.GetComponent<Transform>().eulerAngles, cardThrowTime - 0.3f, 0.0f, cardsRotateGroup, Action.EaseType.EaseIn);
                    player.hand.Add(card);
                    CardDeck.Remove(card);
                }
            }
            IsShuffled = false;
        }, delay);
    }

    void SetupPlayers()
    {
        int activePlayersNumber = 0;
        int targetPlayersNumber = AIPlayersNumber + 1;
        foreach (var player in playerList)
        {
            if (player.GetComponent<Player>().IsIn||player.GetComponent<Player>().IsJoinMiddle)
            {
                activePlayersNumber++;
            }
        }
        if (activePlayersNumber == targetPlayersNumber) 
        {
            return;
        }
        else if (activePlayersNumber > targetPlayersNumber)
        {
            for (int i = activePlayersNumber - 1; i > targetPlayersNumber - 1; i--)
            {
                playerList[i].GetComponent<Player>().LeaveGame();
            }
        }
        else if (activePlayersNumber < targetPlayersNumber)
        {
            for (int i = activePlayersNumber; i < targetPlayersNumber; i++)
            {
                playerList[i].GetComponent<Player>().JoinGame();
            }
        }
    }
    void ResetPlayers()
    {
        foreach(var playerObject in playerList)
        {
            Player player = playerObject.GetComponent<Player>();
            if(player.IsIn)
            {
                player.LeaveGame();
            }
        }
        for (int i = 0; i < AIPlayersNumber + 1; i++)
        {
            playerList[i].GetComponent<Player>().JoinGame();
        }
    }
    void StartNewGame()
    {
        gameState = STATE_IDLE;
        ResetPlayers();
        StartNewRound();
    }

    void StartNewRound()
    {
        gameState = STATE_IDLE;
        cycleCount = 0;
        foreach (var player in playerList)
        {
            player.GetComponent<Player>().ResetForNextRound();
        }
        if(IsRevealing)
        {
            ShuffleCards(1.0f);
        }
        else
        {
            ShuffleCards();
        }
        if(Player1.GetComponent<Player>().IsIn)
        {
            gameState = STATE_PLAYER_TURN;
        }
        StartNewCycle();
    }
    void StartNewCycle()
    {
        gameState = STATE_IDLE;
        cycleCount++;
        foreach(var player in playerList)
        {
            player.GetComponent<Player>().HasActed = false;
        }
        currentPlayerIndex = -1;
        if (IsShuffled && IsRevealing)
        {
            float delay = shuffleTime + 2.0f;
            DealCards(delay);
        }
        else if (IsShuffled && !IsRevealing)
        {
            float delay = shuffleTime + 1.0f;
            DealCards(delay);
        }
        else
        {
            DealCards();
        }
        AdvanceTurn();
    }

    public void AdvanceTurn()
    {
        if (gameState == STATE_IDLE) // Not player's turn
        {
            currentPlayerIndex = (currentPlayerIndex + 1) % playerList.Count;
            Player currentPlayer = playerList[currentPlayerIndex].GetComponent<Player>();
            if (currentPlayer.IsFold || !currentPlayer.IsIn) // Skip the NOT IN || FOLD players
            {
                advanceTurnAttempts++;
                if( advanceTurnAttempts < playerList.Count)
                {
                    AdvanceTurn();
                }
                else
                {
                    advanceTurnAttempts = 0;
                    //Debug.LogError("No eligible players found to take a turn");
                }
            }
            else
            {
                advanceTurnAttempts = 0;

                // Update the game state based on the current player's type
                gameState = currentPlayer.IsAI ? STATE_AI_TURN : STATE_PLAYER_TURN;
                // If it's an AI turn, make the AI take action
                if (currentPlayer.IsAI)
                {
                    // Simulate AI decision-making
                    currentPlayer.AIBehavior();
                }
            }
        }
    }

    public IEnumerator CheckGameFlowState()
    {
        foreach (var playerObj in playerList)
        {
            Player player = playerObj.GetComponent<Player>();
            if(player.IsIn && player.holdingChips.Count == 0) //Kick out looser
            {
                player.LeaveGame();
            }
        }
        if (IsCycleOver())
        {
            if (IsRoundOver())
            {
                DetermineAwardWinner();
                if (IsGameOver())
                {
                    if(isAutoMode)
                    {
                        ToggleAutoMode(false);
                        WriteMenuTelemetryData();
                    }
                    StartNewGame();
                    yield break;
                }
                StartNewRound();
                if (isAutoMode)
                {
                    WriteHandTelemetryData();
                    yield return new WaitForSeconds(3);
                    uiManager.KeyESCCall();
                    yield return new WaitForSeconds(2.01f);
                    uiManager.UITest();
                    yield return new WaitForSeconds(8);
                }
                yield break;
            }
            yield return new WaitForSeconds(0.1f);
            StartNewCycle();
            yield break;
        }
        yield return new WaitForSeconds (0.1f);
        AdvanceTurn();
    }

    // Checking is this 'Cycle' is over.
    // It's for checking am I the last player to play.
    public bool IsCycleOver()
    {
        return playerList.FindAll(player => 
        player.GetComponent<Player>().IsIn && 
        !player.GetComponent<Player>().IsFold && 
        player.GetComponent<Player>().HasActed).Count == 
        playerList.FindAll(player => 
        player.GetComponent<Player>().IsIn && 
        !player.GetComponent<Player>().IsFold).Count;
    }

    // False condition: Bet player is more than 2 && Player already have 5 cards.
    // True check: Is bet player is less than 2 + Is any of player (who is in game) have 5 card
    bool IsRoundOver()
    {
        int activeNonFoldedPlayers = playerList.Count(playerObject => 
        playerObject.GetComponent<Player>().IsIn &&
        !playerObject.GetComponent<Player>().IsFold);
        
        if (activeNonFoldedPlayers < 2) return true;
        
        bool hasPlayerWithFiveCards = playerList.Any(playerObject => 
        playerObject.GetComponent<Player>().IsIn == true && 
        playerObject.GetComponent<Player>().hand.Count == 5);
        if (hasPlayerWithFiveCards) return true;
        
        return false;
    }

    bool IsGameOver()
    {
        if (playerList[0].GetComponent<Player>().IsIn == false) return true;
        int playingPlayersCount = 0;
        foreach(var playerObject in playerList)
        {
            Player player = playerObject.GetComponent<Player>();
            if(player.IsIn)
                playingPlayersCount++;
        }
        if(playingPlayersCount < 2)
        {
            return true;
        }
        else
            return false;
    }
    // Compares two hand values and returns 1 if hand1 is better, -1 if hand2 is better, or 0 if they are equal
    int CompareHands(Vector3 hand1, Vector3 hand2)
    {
        if (hand1.x > hand2.x) return 1; // Higher rank wins
        if (hand1.x < hand2.x) return -1; // Lower rank loses

        // If ranks are equal, compare the primary values (y)
        if (hand1.y > hand2.y) return 1;
        if (hand1.y < hand2.y) return -1;

        // If primary values are equal, optionally compare secondary values (z)
        if (hand1.z > hand2.z) return 1;
        if (hand1.z < hand2.z) return -1;

        return 0; // Hands are equal
    }
    void DetermineAwardWinner()
    {
        if (playerList == null || playerList.Count == 0)
        {
            return;
        }
        for(int i = 0; i < playerList.Count; i++)
        {
            if (playerList[i] == null)
            {
                Debug.LogWarning($"Player at index {i} is null");
                continue;
            }
            var playerComponent = playerList[i].GetComponent<Player>();
            if (playerComponent == null)
            {
                Debug.LogWarning($"Player component at index {i} is missing.");
                continue;
            }

            playerComponent.IsWin = false;
        }

        int activeNonFoldedPlayers = playerList.Count(playerObject => 
        {
            var player = playerObject?.GetComponent<Player>();
            return player != null && player.IsIn && !player.IsFold;
        });
        // playerObject.GetComponent<Player>().IsIn && 
        // !playerObject.GetComponent<Player>().IsFold);

        // If all but one have folded, that player is the winner
        if (activeNonFoldedPlayers <= 1)
        {
            Player winnerBeforeEvaluation = playerList
            .Select(playerObject => playerObject?.GetComponent<Player>())
            .FirstOrDefault(player => player != null && player.IsIn && !player.IsFold);
            // Player winnerBeforeEvaluation = playerList.FirstOrDefault(playerObject => 
            // playerObject.GetComponent<Player>().IsIn && 
            // !playerObject.GetComponent<Player>().IsFold).GetComponent<Player>();

            if (winnerBeforeEvaluation != null)
            {
                Debug.Log(winnerBeforeEvaluation.Name + " wins by default as all others have folded.");
                Awards(winnerBeforeEvaluation);
                return;
            }
        }

        Player winner = null;
        Vector3 highestHandValue = Vector3.zero;

        foreach (var playerObject in playerList)
        {
            Player player = playerObject.GetComponent<Player>();
            if (player.IsIn && !player.IsFold)
            {
                player.EvaluateHands(); // Ensure hands are evaluated

                // Check if this player's hand is better than the current best hand
                if (winner == null || CompareHands(player.handsValue, highestHandValue) > 0)
                {
                    winner = player;
                    highestHandValue = player.handsValue;
                }
            }
        }

        if (winner != null)
        {
            Debug.Log(winner.name + " wins with a hand value of " + highestHandValue);
            Awards(winner);
        }
        else
        {
            Debug.Log("No winner this round.");
        }
    }
    void Awards(Player winner)
    {
        winner.IsWin = true;
        IsRevealing = true;
        float delayBetween = 1.75f;
        int chipPotCount = ChipPot.Count;
        for (int i = 0; i < chipPotCount; i++)
        {
            winner.StackChip(ChipPot, ChipPot[^1], delayBetween);
            delayBetween += 0.05f; // Max 1.5 + 1.75f
        }
        Debug.Log(winner.name + " wins ");
        foreach (var player in playerList)
        {
            player.GetComponent<Player>().RevealCards();
        }
    }
    public void WriteMenuTelemetryData()
    {
        System.Text.StringBuilder telemetryLine = new ();
        DataStream.WriteLine("Resume,Graphic,Audio,Resolution,Exit,High,Medium,Low");
        for (int i = 0; i < uiManager.MainOptionsList.Count; i++)
        {
            if (i == 0)
            {
                ResumeButton mainButton = uiManager.MainOptionsList[0].GetComponent<ResumeButton>();
                string pressedCount = mainButton.buttonPressed.ToString();
                telemetryLine.Append(pressedCount + ",");
            }
            else if (i == 4)
            {
                ExitButton mainButton = uiManager.MainOptionsList[4].GetComponent<ExitButton>();
                string pressedCount = mainButton.buttonPressed.ToString();
                telemetryLine.Append(pressedCount + ",");
            }
            else
            {
                UIButton mainButton = uiManager.MainOptionsList[i].GetComponent<UIButton>();
                string pressedCount = mainButton.buttonPressed.ToString();
                telemetryLine.Append(pressedCount + ",");
            }
        }
        for (int i = 0; i < 3; i++)
        {
            UIButton subButton = uiManager.SubOptionsList[i].GetComponent<UIButton>();
            string pressedCount = subButton.buttonPressed.ToString();
            telemetryLine.Append(pressedCount);
            if(i < uiManager.SubOptionsList.Count -1)
            {
                telemetryLine.Append(",");
            }
        }
        DataStream.WriteLine(telemetryLine.ToString());
        DataStream.Flush();
    }
    public void WriteHandTelemetryData()
    {
        System.Text.StringBuilder telemetryLine = new ();
        for(int i = 0; i < playerList.Count; i++)
        {
            Player player = playerList[i].GetComponent<Player>();
            string chipCount = player.holdingChips.Count.ToString();
            string result = player.IsFold ? "F" : player.IsWin ? "W" : "L";
            telemetryLine.Append(chipCount + result);

            // If not the last player, add a comma for separation
            if (i < playerList.Count - 1)
            {
                telemetryLine.Append(",");
            }
        }
        DataStream.WriteLine(telemetryLine.ToString());
        DataStream.Flush();
    }
    private void OnDrawGizmos() // Draw Gizmo for placeholder
    {
        Gizmos.color = GizmoColor;

        Gizmos.DrawSphere(Player1.transform.position, 0.1f);
        Gizmos.DrawSphere(Player2.transform.position, 0.1f);
        Gizmos.DrawSphere(Player3.transform.position, 0.1f);
        Gizmos.DrawSphere(Player4.transform.position, 0.1f);
        Gizmos.DrawSphere(Player5.transform.position, 0.1f);
        Gizmos.DrawSphere(Player6.transform.position, 0.1f);
        CardDeckPosition = transform.Find("CardDeckSpot").position;
        Gizmos.DrawSphere(CardDeckPosition, 0.1f);
    }
    void Awake()
    {
        actionList = GetComponent<ActionList>(); // Assign the reference to the ActionList component
        uiManager = GameObject.Find("UIManager").GetComponent<UIManager>(); // Assign the reference to the ActionList component
        buttonActionList = GameObject.Find("Buttons").GetComponent<ActionList>(); // Assign the reference to the ActionList component
        playerList.Add(GameObject.Find("Player1"));
        playerList.Add(GameObject.Find("Player2"));
        playerList.Add(GameObject.Find("Player3"));
        playerList.Add(GameObject.Find("Player4"));
        playerList.Add(GameObject.Find("Player5"));
        playerList.Add(GameObject.Find("Player6"));

        for (int i = 0; i < AIPlayersNumber + 1; i++)
        {
            playerList[i].GetComponent<Player>().IsIn = true;
        }
    }

    void Start()
    {
        DataStream = new StreamWriter("TelemetryData.csv", false);
        DataStream.WriteLine("You,Random Rick,Crazy Carl,Bluffy Ben,Scary Suzie,Smart Sam");
        InitializeDeck();
        StartNewGame();
    }

    private Coroutine autoModeCoroutine;
    void ToggleAutoMode(bool setTo)
    {
        if((setTo && actionList.Actions.Count <= 0) || !setTo)
        {
            isAutoMode = setTo;
            Time.timeScale = isAutoMode ? 5.0f : 1.0f;

            Player player = playerList[0].GetComponent<Player>();
            player.Name = setTo ? "Smart Sam" : "You";
            if(setTo) DataStream.WriteLine("Smart Sam,Random Rick,Crazy Carl,Bluffy Ben,Scary Suzie,Smart Sam");
            else DataStream.WriteLine("You,Random Rick,Crazy Carl,Bluffy Ben,Scary Suzie,Smart Sam");
            player.nameTMPro.GetComponent<TextMeshPro>().text = player.Name;
        
            autoModeCoroutine = StartCoroutine(AutoModeRoutine());
        }
    }
    IEnumerator AutoModeRoutine() 
    {
        while (isAutoMode)
        {
            Player currentPlayer = playerList[currentPlayerIndex].GetComponent<Player>();
            currentPlayer.AIBehavior(); // Let the AI decide on bet or fold
            yield return new WaitForSeconds(4); // Wait for a bit before the next action
            
            yield return null; // Ensure the loop doesn't block the game
        }
    }
    void Update()
    {
        if(!isUIon && !isAutoMode)
        {
            // Testing Key Index
            // ESC  : UI On/Off
            // 1~5  : Changing AI number
            // A    : AutoMode Toggle

            if(Input.GetKeyDown(KeyCode.Escape))
            {
                actionList.isPause = !actionList.isPause;
            }

            if(Input.GetKeyDown(KeyCode.A))
            {
                ToggleAutoMode(true);
            }
            // Player number changing
            if(Input.GetKeyDown(KeyCode.Alpha1))
            {
                AIPlayersNumber = 1;
                SetupPlayers();
            }
            if(Input.GetKeyDown(KeyCode.Alpha2))
            {
                AIPlayersNumber = 2;
                SetupPlayers();
            }
            if(Input.GetKeyDown(KeyCode.Alpha3))
            {
                AIPlayersNumber = 3;
                SetupPlayers();
            }
            if(Input.GetKeyDown(KeyCode.Alpha4))
            {
                AIPlayersNumber = 4;
                SetupPlayers();
            }
            if(Input.GetKeyDown(KeyCode.Alpha5))
            {
                AIPlayersNumber = 5;
                SetupPlayers();
            }
            if(Input.GetKeyDown(KeyCode.Alpha6))
            {
                ResetPlayers();
            }
        }
        else
        {
            if(Input.GetKeyDown(KeyCode.A))
            {
                ToggleAutoMode(false);
            }
        }
    }
} // public class GameManager : MonoBehaviour
} // namespace PokerTable